<?php
require_once __DIR__ . '/db.php';

function sanitize($str) {
    return htmlspecialchars(trim($str), ENT_QUOTES, 'UTF-8');
}

function redirect($url) {
    header('Location: ' . $url);
    exit;
}

function isLoggedIn() {
    return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
}

function requireLogin() {
    if (!isLoggedIn()) {
        redirect('login.php');
    }
}

function uploadImage($file, $targetDir = 'uploads/products/') {
    $baseDir = __DIR__ . '/../' . $targetDir;
    if (!is_dir($baseDir)) {
        mkdir($baseDir, 0755, true);
    }

    $allowed = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
    if (!in_array($file['type'], $allowed)) {
        return ['error' => 'Invalid file type. Only JPG, PNG, GIF, WEBP allowed.'];
    }

    if ($file['size'] > 5 * 1024 * 1024) {
        return ['error' => 'File too large. Maximum 5MB.'];
    }

    $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = uniqid('img_') . '.' . strtolower($ext);
    $destination = $baseDir . $filename;

    if (move_uploaded_file($file['tmp_name'], $destination)) {
        return ['path' => $targetDir . $filename];
    }

    return ['error' => 'Upload failed.'];
}

function deleteImage($path) {
    $fullPath = __DIR__ . '/../' . $path;
    if (file_exists($fullPath) && strpos($path, 'uploads/') === 0) {
        unlink($fullPath);
    }
}

function getCategories() {
    return fetchAll("SELECT * FROM categories ORDER BY sort_order, name");
}

function getSubcategories($categoryId = null) {
    if ($categoryId) {
        return fetchAll("SELECT s.*, c.name as category_name FROM subcategories s JOIN categories c ON s.category_id = c.id WHERE s.category_id = ? ORDER BY s.sort_order, s.name", [$categoryId]);
    }
    return fetchAll("SELECT s.*, c.name as category_name FROM subcategories s JOIN categories c ON s.category_id = c.id ORDER BY c.sort_order, s.sort_order, s.name");
}

function getProducts($subcategoryId = null, $categoryId = null) {
    if ($subcategoryId) {
        return fetchAll("SELECT p.*, s.name as subcategory_name, c.name as category_name FROM products p JOIN subcategories s ON p.subcategory_id = s.id JOIN categories c ON s.category_id = c.id WHERE p.subcategory_id = ? ORDER BY p.sort_order, p.name", [$subcategoryId]);
    }
    if ($categoryId) {
        return fetchAll("SELECT p.*, s.name as subcategory_name, c.name as category_name FROM products p JOIN subcategories s ON p.subcategory_id = s.id JOIN categories c ON s.category_id = c.id WHERE s.category_id = ? ORDER BY s.sort_order, p.sort_order, p.name", [$categoryId]);
    }
    return fetchAll("SELECT p.*, s.name as subcategory_name, c.name as category_name FROM products p JOIN subcategories s ON p.subcategory_id = s.id JOIN categories c ON s.category_id = c.id ORDER BY c.sort_order, s.sort_order, p.sort_order, p.name");
}

function getProductImages($productId) {
    return fetchAll("SELECT * FROM product_images WHERE product_id = ? ORDER BY sort_order", [$productId]);
}

function flash($key, $message = null) {
    if ($message !== null) {
        $_SESSION['flash_' . $key] = $message;
    } else {
        $msg = $_SESSION['flash_' . $key] ?? null;
        unset($_SESSION['flash_' . $key]);
        return $msg;
    }
}
