<?php
session_start();
header('Content-Type: application/json');

if (!file_exists(__DIR__ . '/config.php')) {
    http_response_code(500);
    echo json_encode(['message' => 'config.php not found. Please rename config.sample.php to config.php and fill in your database credentials.']);
    exit;
}

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/includes/db.php';

$method = $_SERVER['REQUEST_METHOD'];
$uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);

$scriptDir = dirname($_SERVER['SCRIPT_NAME']);
if ($scriptDir !== '/' && $scriptDir !== '\\') {
    $uri = substr($uri, strlen($scriptDir));
}
$uri = '/' . ltrim($uri, '/');

function jsonResponse($data, $code = 200) {
    http_response_code($code);
    echo json_encode($data);
    exit;
}

function getJsonBody() {
    $body = file_get_contents('php://input');
    return json_decode($body, true) ?: [];
}

function requireAuth() {
    if (empty($_SESSION['userId'])) {
        jsonResponse(['message' => 'Unauthorized'], 401);
    }
}

function handleUpload() {
    requireAuth();
    if (empty($_FILES['file'])) {
        jsonResponse(['message' => 'No file uploaded'], 400);
    }

    $file = $_FILES['file'];
    $allowed = ['image/jpeg', 'image/png', 'image/gif', 'image/webp', 'image/svg+xml'];
    if (!in_array($file['type'], $allowed)) {
        jsonResponse(['message' => 'Invalid file type'], 400);
    }

    if ($file['size'] > UPLOAD_MAX_SIZE) {
        jsonResponse(['message' => 'File too large'], 400);
    }

    $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = time() . '-' . bin2hex(random_bytes(8)) . '.' . $ext;
    $uploadDir = __DIR__ . '/uploads/';

    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }

    if (!move_uploaded_file($file['tmp_name'], $uploadDir . $filename)) {
        jsonResponse(['message' => 'Upload failed'], 500);
    }

    jsonResponse(['url' => '/uploads/' . $filename]);
}

// ===== ROUTING =====

// Install endpoints
if ($uri === '/api/install/status' && $method === 'GET') {
    try {
        if (!tableExists('install_status')) {
            jsonResponse(['installed' => false]);
        }
        $db = getDB();
        $stmt = $db->query("SELECT installed FROM install_status LIMIT 1");
        $row = $stmt->fetch();
        jsonResponse(['installed' => $row ? (bool)$row['installed'] : false]);
    } catch (Exception $e) {
        jsonResponse(['installed' => false]);
    }
}

if ($uri === '/api/install' && $method === 'POST') {
    try {
        $body = getJsonBody();
        
        // Check if already installed
        if (tableExists('install_status')) {
            $db = getDB();
            $stmt = $db->query("SELECT installed FROM install_status LIMIT 1");
            $row = $stmt->fetch();
            if ($row && $row['installed']) {
                jsonResponse(['message' => 'Already installed'], 400);
            }
        }

        if (empty($body['adminUsername']) || empty($body['adminPassword']) || empty($body['adminEmail'])) {
            jsonResponse(['message' => 'Admin credentials are required'], 400);
        }

        $db = getDB();

        // Create all tables
        $db->exec("
            CREATE TABLE IF NOT EXISTS users (
                id INT AUTO_INCREMENT PRIMARY KEY,
                username VARCHAR(255) NOT NULL UNIQUE,
                password VARCHAR(255) NOT NULL,
                email VARCHAR(255),
                role VARCHAR(50) NOT NULL DEFAULT 'admin'
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

            CREATE TABLE IF NOT EXISTS site_settings (
                id INT AUTO_INCREMENT PRIMARY KEY,
                site_title VARCHAR(255) NOT NULL DEFAULT 'RIG & DIG',
                tagline TEXT,
                logo_url TEXT,
                contact_email VARCHAR(255),
                contact_phone VARCHAR(100),
                contact_address TEXT,
                hero_title VARCHAR(255) DEFAULT 'Built for the Toughest Jobs',
                hero_subtitle TEXT,
                about_text TEXT,
                footer_text TEXT
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

            CREATE TABLE IF NOT EXISTS fleet_items (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(255) NOT NULL,
                category VARCHAR(255) NOT NULL,
                description TEXT,
                image_url TEXT,
                specs TEXT,
                available TINYINT(1) NOT NULL DEFAULT 1,
                sort_order INT DEFAULT 0
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

            CREATE TABLE IF NOT EXISTS services (
                id INT AUTO_INCREMENT PRIMARY KEY,
                title VARCHAR(255) NOT NULL,
                description TEXT NOT NULL,
                icon VARCHAR(100),
                featured TINYINT(1) NOT NULL DEFAULT 0,
                sort_order INT DEFAULT 0
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

            CREATE TABLE IF NOT EXISTS products (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(255) NOT NULL,
                category VARCHAR(255) NOT NULL,
                description TEXT,
                image_url TEXT,
                specs TEXT,
                price VARCHAR(255),
                in_stock TINYINT(1) NOT NULL DEFAULT 1,
                featured TINYINT(1) NOT NULL DEFAULT 0,
                sort_order INT DEFAULT 0
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

            CREATE TABLE IF NOT EXISTS leads (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(255) NOT NULL,
                email VARCHAR(255) NOT NULL,
                phone VARCHAR(100),
                company VARCHAR(255),
                message TEXT NOT NULL,
                `read` TINYINT(1) NOT NULL DEFAULT 0,
                created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

            CREATE TABLE IF NOT EXISTS install_status (
                id INT AUTO_INCREMENT PRIMARY KEY,
                installed TINYINT(1) NOT NULL DEFAULT 0,
                installed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
        ");

        // Create admin user
        $hashedPassword = password_hash($body['adminPassword'], PASSWORD_BCRYPT, ['cost' => 12]);
        $stmt = $db->prepare("INSERT INTO users (username, password, email, role) VALUES (?, ?, ?, 'admin')");
        $stmt->execute([$body['adminUsername'], $hashedPassword, $body['adminEmail']]);

        // Create site settings
        $stmt = $db->prepare("INSERT INTO site_settings (site_title, contact_email, contact_phone, tagline, hero_title, hero_subtitle) VALUES (?, ?, ?, 'Heavy Equipment & Professional Services', 'Built for the Toughest Jobs', 'Professional heavy equipment services for construction, mining, and infrastructure projects.')");
        $stmt->execute([
            $body['siteTitle'] ?? 'RIG & DIG',
            $body['contactEmail'] ?? null,
            $body['contactPhone'] ?? null,
        ]);

        // Mark as installed
        $db->exec("INSERT INTO install_status (installed) VALUES (1)");

        jsonResponse(['success' => true, 'message' => 'Installation complete']);
    } catch (Exception $e) {
        jsonResponse(['message' => $e->getMessage()], 500);
    }
}

// Auth endpoints
if ($uri === '/api/auth/login' && $method === 'POST') {
    $body = getJsonBody();
    if (empty($body['username']) || empty($body['password'])) {
        jsonResponse(['message' => 'Username and password are required'], 400);
    }

    try {
        $db = getDB();
        $stmt = $db->prepare("SELECT * FROM users WHERE username = ?");
        $stmt->execute([$body['username']]);
        $user = $stmt->fetch();

        if (!$user || !password_verify($body['password'], $user['password'])) {
            jsonResponse(['message' => 'Invalid credentials'], 401);
        }

        $_SESSION['userId'] = $user['id'];
        jsonResponse(['id' => $user['id'], 'username' => $user['username'], 'role' => $user['role']]);
    } catch (Exception $e) {
        jsonResponse(['message' => 'Login failed'], 500);
    }
}

if ($uri === '/api/auth/me' && $method === 'GET') {
    if (empty($_SESSION['userId'])) {
        jsonResponse(['message' => 'Not authenticated'], 401);
    }
    try {
        $db = getDB();
        $stmt = $db->prepare("SELECT id, username, role FROM users WHERE id = ?");
        $stmt->execute([$_SESSION['userId']]);
        $user = $stmt->fetch();
        if (!$user) {
            jsonResponse(['message' => 'User not found'], 401);
        }
        jsonResponse($user);
    } catch (Exception $e) {
        jsonResponse(['message' => 'Error'], 500);
    }
}

if ($uri === '/api/auth/logout' && $method === 'POST') {
    session_destroy();
    jsonResponse(['success' => true]);
}

// Public: Settings
if ($uri === '/api/settings' && $method === 'GET') {
    try {
        $db = getDB();
        $stmt = $db->query("SELECT * FROM site_settings LIMIT 1");
        $row = $stmt->fetch();
        if ($row) {
            $row = mapSettings($row);
        }
        jsonResponse($row ?: new stdClass());
    } catch (Exception $e) {
        jsonResponse(new stdClass());
    }
}

// Public: Fleet
if ($uri === '/api/fleet' && $method === 'GET') {
    try {
        $db = getDB();
        $stmt = $db->query("SELECT * FROM fleet_items ORDER BY sort_order ASC");
        $rows = $stmt->fetchAll();
        foreach ($rows as &$r) {
            $r['available'] = (bool)$r['available'];
            $r['sortOrder'] = (int)$r['sort_order'];
            $r['imageUrl'] = $r['image_url'];
            unset($r['sort_order'], $r['image_url']);
        }
        jsonResponse($rows);
    } catch (Exception $e) {
        jsonResponse([]);
    }
}

// Public: Services
if ($uri === '/api/services' && $method === 'GET') {
    try {
        $db = getDB();
        $stmt = $db->query("SELECT * FROM services ORDER BY sort_order ASC");
        $rows = $stmt->fetchAll();
        foreach ($rows as &$r) {
            $r['featured'] = (bool)$r['featured'];
            $r['sortOrder'] = (int)$r['sort_order'];
            unset($r['sort_order']);
        }
        jsonResponse($rows);
    } catch (Exception $e) {
        jsonResponse([]);
    }
}

// Public: Products
if ($uri === '/api/products' && $method === 'GET') {
    try {
        $db = getDB();
        $stmt = $db->query("SELECT * FROM products ORDER BY sort_order ASC");
        $rows = $stmt->fetchAll();
        foreach ($rows as &$r) {
            $r['inStock'] = (bool)$r['in_stock'];
            $r['featured'] = (bool)$r['featured'];
            $r['sortOrder'] = (int)$r['sort_order'];
            $r['imageUrl'] = $r['image_url'];
            unset($r['sort_order'], $r['in_stock'], $r['image_url']);
        }
        jsonResponse($rows);
    } catch (Exception $e) {
        jsonResponse([]);
    }
}

// Public: Submit lead
if ($uri === '/api/leads' && $method === 'POST') {
    $body = getJsonBody();
    if (empty($body['name']) || empty($body['email']) || empty($body['message'])) {
        jsonResponse(['message' => 'Name, email, and message are required'], 400);
    }
    try {
        $db = getDB();
        $stmt = $db->prepare("INSERT INTO leads (name, email, phone, company, message) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([
            $body['name'],
            $body['email'],
            $body['phone'] ?? null,
            $body['company'] ?? null,
            $body['message'],
        ]);
        $id = $db->lastInsertId();
        $stmt = $db->prepare("SELECT * FROM leads WHERE id = ?");
        $stmt->execute([$id]);
        $lead = $stmt->fetch();
        $lead['read'] = (bool)$lead['read'];
        $lead['createdAt'] = $lead['created_at'];
        unset($lead['created_at']);
        jsonResponse($lead, 201);
    } catch (Exception $e) {
        jsonResponse(['message' => 'Failed to submit'], 400);
    }
}

// Admin: Upload
if ($uri === '/api/admin/upload' && $method === 'POST') {
    handleUpload();
}

// Admin: Settings
if ($uri === '/api/admin/settings' && $method === 'PATCH') {
    requireAuth();
    $body = getJsonBody();
    try {
        $db = getDB();
        $stmt = $db->query("SELECT id FROM site_settings LIMIT 1");
        $existing = $stmt->fetch();

        $fields = ['site_title', 'tagline', 'logo_url', 'contact_email', 'contact_phone', 'contact_address', 'hero_title', 'hero_subtitle', 'about_text', 'footer_text'];
        $camelMap = [
            'siteTitle' => 'site_title', 'tagline' => 'tagline', 'logoUrl' => 'logo_url',
            'contactEmail' => 'contact_email', 'contactPhone' => 'contact_phone',
            'contactAddress' => 'contact_address', 'heroTitle' => 'hero_title',
            'heroSubtitle' => 'hero_subtitle', 'aboutText' => 'about_text', 'footerText' => 'footer_text',
        ];

        $updates = [];
        $values = [];
        foreach ($camelMap as $camel => $snake) {
            if (array_key_exists($camel, $body)) {
                $updates[] = "$snake = ?";
                $values[] = $body[$camel];
            }
        }

        if (!empty($updates) && $existing) {
            $values[] = $existing['id'];
            $stmt = $db->prepare("UPDATE site_settings SET " . implode(', ', $updates) . " WHERE id = ?");
            $stmt->execute($values);
        }

        $stmt = $db->query("SELECT * FROM site_settings LIMIT 1");
        $settings = $stmt->fetch();
        jsonResponse(mapSettings($settings));
    } catch (Exception $e) {
        jsonResponse(['message' => $e->getMessage()], 500);
    }
}

// Admin: Fleet CRUD
if (preg_match('#^/api/admin/fleet(/(\d+))?$#', $uri, $m)) {
    requireAuth();
    $id = $m[2] ?? null;

    if ($method === 'POST') {
        $body = getJsonBody();
        try {
            $db = getDB();
            $stmt = $db->prepare("INSERT INTO fleet_items (name, category, description, image_url, specs, available, sort_order) VALUES (?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([
                $body['name'], $body['category'], $body['description'] ?? null,
                $body['imageUrl'] ?? null, $body['specs'] ?? null,
                isset($body['available']) ? ($body['available'] ? 1 : 0) : 1,
                $body['sortOrder'] ?? 0,
            ]);
            $newId = $db->lastInsertId();
            $stmt = $db->prepare("SELECT * FROM fleet_items WHERE id = ?");
            $stmt->execute([$newId]);
            $item = $stmt->fetch();
            $item = mapFleetItem($item);
            jsonResponse($item, 201);
        } catch (Exception $e) {
            jsonResponse(['message' => $e->getMessage()], 400);
        }
    }

    if ($method === 'PATCH' && $id) {
        $body = getJsonBody();
        try {
            $db = getDB();
            $updates = [];
            $values = [];
            $fieldMap = [
                'name' => 'name', 'category' => 'category', 'description' => 'description',
                'imageUrl' => 'image_url', 'specs' => 'specs', 'sortOrder' => 'sort_order',
            ];
            foreach ($fieldMap as $camel => $snake) {
                if (array_key_exists($camel, $body)) {
                    $updates[] = "$snake = ?";
                    $values[] = $body[$camel];
                }
            }
            if (array_key_exists('available', $body)) {
                $updates[] = "available = ?";
                $values[] = $body['available'] ? 1 : 0;
            }
            if (!empty($updates)) {
                $values[] = $id;
                $stmt = $db->prepare("UPDATE fleet_items SET " . implode(', ', $updates) . " WHERE id = ?");
                $stmt->execute($values);
            }
            $stmt = $db->prepare("SELECT * FROM fleet_items WHERE id = ?");
            $stmt->execute([$id]);
            $item = $stmt->fetch();
            if (!$item) jsonResponse(['message' => 'Not found'], 404);
            jsonResponse(mapFleetItem($item));
        } catch (Exception $e) {
            jsonResponse(['message' => $e->getMessage()], 500);
        }
    }

    if ($method === 'DELETE' && $id) {
        try {
            $db = getDB();
            $stmt = $db->prepare("DELETE FROM fleet_items WHERE id = ?");
            $stmt->execute([$id]);
            jsonResponse(['success' => true]);
        } catch (Exception $e) {
            jsonResponse(['message' => $e->getMessage()], 500);
        }
    }
}

// Admin: Services CRUD
if (preg_match('#^/api/admin/services(/(\d+))?$#', $uri, $m)) {
    requireAuth();
    $id = $m[2] ?? null;

    if ($method === 'POST') {
        $body = getJsonBody();
        try {
            $db = getDB();
            $stmt = $db->prepare("INSERT INTO services (title, description, icon, featured, sort_order) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([
                $body['title'], $body['description'],
                $body['icon'] ?? null,
                isset($body['featured']) ? ($body['featured'] ? 1 : 0) : 0,
                $body['sortOrder'] ?? 0,
            ]);
            $newId = $db->lastInsertId();
            $stmt = $db->prepare("SELECT * FROM services WHERE id = ?");
            $stmt->execute([$newId]);
            $item = $stmt->fetch();
            $item = mapService($item);
            jsonResponse($item, 201);
        } catch (Exception $e) {
            jsonResponse(['message' => $e->getMessage()], 400);
        }
    }

    if ($method === 'PATCH' && $id) {
        $body = getJsonBody();
        try {
            $db = getDB();
            $updates = [];
            $values = [];
            $fieldMap = ['title' => 'title', 'description' => 'description', 'icon' => 'icon', 'sortOrder' => 'sort_order'];
            foreach ($fieldMap as $camel => $snake) {
                if (array_key_exists($camel, $body)) {
                    $updates[] = "$snake = ?";
                    $values[] = $body[$camel];
                }
            }
            if (array_key_exists('featured', $body)) {
                $updates[] = "featured = ?";
                $values[] = $body['featured'] ? 1 : 0;
            }
            if (!empty($updates)) {
                $values[] = $id;
                $stmt = $db->prepare("UPDATE services SET " . implode(', ', $updates) . " WHERE id = ?");
                $stmt->execute($values);
            }
            $stmt = $db->prepare("SELECT * FROM services WHERE id = ?");
            $stmt->execute([$id]);
            $item = $stmt->fetch();
            if (!$item) jsonResponse(['message' => 'Not found'], 404);
            jsonResponse(mapService($item));
        } catch (Exception $e) {
            jsonResponse(['message' => $e->getMessage()], 500);
        }
    }

    if ($method === 'DELETE' && $id) {
        try {
            $db = getDB();
            $stmt = $db->prepare("DELETE FROM services WHERE id = ?");
            $stmt->execute([$id]);
            jsonResponse(['success' => true]);
        } catch (Exception $e) {
            jsonResponse(['message' => $e->getMessage()], 500);
        }
    }
}

// Admin: Products CRUD
if (preg_match('#^/api/admin/products(/(\d+))?$#', $uri, $m)) {
    requireAuth();
    $id = $m[2] ?? null;

    if ($method === 'POST') {
        $body = getJsonBody();
        try {
            $db = getDB();
            $stmt = $db->prepare("INSERT INTO products (name, category, description, image_url, specs, price, in_stock, featured, sort_order) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([
                $body['name'], $body['category'],
                $body['description'] ?? null, $body['imageUrl'] ?? null,
                $body['specs'] ?? null, $body['price'] ?? null,
                isset($body['inStock']) ? ($body['inStock'] ? 1 : 0) : 1,
                isset($body['featured']) ? ($body['featured'] ? 1 : 0) : 0,
                $body['sortOrder'] ?? 0,
            ]);
            $newId = $db->lastInsertId();
            $stmt = $db->prepare("SELECT * FROM products WHERE id = ?");
            $stmt->execute([$newId]);
            $item = $stmt->fetch();
            jsonResponse(mapProduct($item), 201);
        } catch (Exception $e) {
            jsonResponse(['message' => $e->getMessage()], 400);
        }
    }

    if ($method === 'PATCH' && $id) {
        $body = getJsonBody();
        try {
            $db = getDB();
            $updates = [];
            $values = [];
            $fieldMap = [
                'name' => 'name', 'category' => 'category', 'description' => 'description',
                'imageUrl' => 'image_url', 'specs' => 'specs', 'price' => 'price', 'sortOrder' => 'sort_order',
            ];
            foreach ($fieldMap as $camel => $snake) {
                if (array_key_exists($camel, $body)) {
                    $updates[] = "$snake = ?";
                    $values[] = $body[$camel];
                }
            }
            if (array_key_exists('inStock', $body)) {
                $updates[] = "in_stock = ?";
                $values[] = $body['inStock'] ? 1 : 0;
            }
            if (array_key_exists('featured', $body)) {
                $updates[] = "featured = ?";
                $values[] = $body['featured'] ? 1 : 0;
            }
            if (!empty($updates)) {
                $values[] = $id;
                $stmt = $db->prepare("UPDATE products SET " . implode(', ', $updates) . " WHERE id = ?");
                $stmt->execute($values);
            }
            $stmt = $db->prepare("SELECT * FROM products WHERE id = ?");
            $stmt->execute([$id]);
            $item = $stmt->fetch();
            if (!$item) jsonResponse(['message' => 'Not found'], 404);
            jsonResponse(mapProduct($item));
        } catch (Exception $e) {
            jsonResponse(['message' => $e->getMessage()], 500);
        }
    }

    if ($method === 'DELETE' && $id) {
        try {
            $db = getDB();
            $stmt = $db->prepare("DELETE FROM products WHERE id = ?");
            $stmt->execute([$id]);
            jsonResponse(['success' => true]);
        } catch (Exception $e) {
            jsonResponse(['message' => $e->getMessage()], 500);
        }
    }
}

// Admin: Leads
if ($uri === '/api/admin/leads' && $method === 'GET') {
    requireAuth();
    try {
        $db = getDB();
        $stmt = $db->query("SELECT * FROM leads ORDER BY created_at DESC");
        $rows = $stmt->fetchAll();
        foreach ($rows as &$r) {
            $r = mapLead($r);
        }
        jsonResponse($rows);
    } catch (Exception $e) {
        jsonResponse(['message' => $e->getMessage()], 500);
    }
}

if (preg_match('#^/api/admin/leads/(\d+)$#', $uri, $m)) {
    requireAuth();
    $id = $m[1];

    if ($method === 'PATCH') {
        $body = getJsonBody();
        try {
            $db = getDB();
            $updates = [];
            $values = [];
            if (array_key_exists('read', $body)) {
                $updates[] = "`read` = ?";
                $values[] = $body['read'] ? 1 : 0;
            }
            if (!empty($updates)) {
                $values[] = $id;
                $stmt = $db->prepare("UPDATE leads SET " . implode(', ', $updates) . " WHERE id = ?");
                $stmt->execute($values);
            }
            $stmt = $db->prepare("SELECT * FROM leads WHERE id = ?");
            $stmt->execute([$id]);
            $lead = $stmt->fetch();
            if (!$lead) jsonResponse(['message' => 'Not found'], 404);
            jsonResponse(mapLead($lead));
        } catch (Exception $e) {
            jsonResponse(['message' => $e->getMessage()], 500);
        }
    }

    if ($method === 'DELETE') {
        try {
            $db = getDB();
            $stmt = $db->prepare("DELETE FROM leads WHERE id = ?");
            $stmt->execute([$id]);
            jsonResponse(['success' => true]);
        } catch (Exception $e) {
            jsonResponse(['message' => $e->getMessage()], 500);
        }
    }
}

// Helper: map database rows to camelCase for frontend
function mapFleetItem($row) {
    if (!$row) return $row;
    return [
        'id' => (int)$row['id'],
        'name' => $row['name'],
        'category' => $row['category'],
        'description' => $row['description'],
        'imageUrl' => $row['image_url'],
        'specs' => $row['specs'],
        'available' => (bool)$row['available'],
        'sortOrder' => (int)$row['sort_order'],
    ];
}

function mapService($row) {
    if (!$row) return $row;
    return [
        'id' => (int)$row['id'],
        'title' => $row['title'],
        'description' => $row['description'],
        'icon' => $row['icon'],
        'featured' => (bool)$row['featured'],
        'sortOrder' => (int)$row['sort_order'],
    ];
}

function mapProduct($row) {
    if (!$row) return $row;
    return [
        'id' => (int)$row['id'],
        'name' => $row['name'],
        'category' => $row['category'],
        'description' => $row['description'],
        'imageUrl' => $row['image_url'],
        'specs' => $row['specs'],
        'price' => $row['price'],
        'inStock' => (bool)$row['in_stock'],
        'featured' => (bool)$row['featured'],
        'sortOrder' => (int)$row['sort_order'],
    ];
}

function mapLead($row) {
    if (!$row) return $row;
    return [
        'id' => (int)$row['id'],
        'name' => $row['name'],
        'email' => $row['email'],
        'phone' => $row['phone'],
        'company' => $row['company'],
        'message' => $row['message'],
        'read' => (bool)$row['read'],
        'createdAt' => $row['created_at'],
    ];
}

function mapSettings($row) {
    if (!$row) return $row;
    return [
        'id' => (int)$row['id'],
        'siteTitle' => $row['site_title'],
        'tagline' => $row['tagline'],
        'logoUrl' => $row['logo_url'],
        'contactEmail' => $row['contact_email'],
        'contactPhone' => $row['contact_phone'],
        'contactAddress' => $row['contact_address'],
        'heroTitle' => $row['hero_title'],
        'heroSubtitle' => $row['hero_subtitle'],
        'aboutText' => $row['about_text'],
        'footerText' => $row['footer_text'],
    ];
}

// If no route matched
http_response_code(404);
echo json_encode(['message' => 'Not found']);
